/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2025 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class Container implements Closeable {

    static class ContainerThreadGroup extends ThreadGroup implements ThreadFactory {
        Container container;
        public ContainerThreadGroup(String name, Container acontainer) {
            super(name);
            container = acontainer;
        }
        @Override
        public Thread newThread(Runnable r) {
            return new Thread(this, r, "ActionThread-"+Integer.toString(container.id()));
        }
    }

    int id;
    ContainerThreadGroup threadGroup;
    ThreadPoolExecutor actionThreads;
    ClientClassLoader clientClassLoader = new ClientClassLoader(this);
    HashMap<String,Action> actions = new HashMap<String,Action>();
    ArrayList<Closeable> cleanups = new ArrayList<Closeable>();
    HashMap<String, ClassCodeTransformer> transformers = new HashMap<String, ClassCodeTransformer>();
    Log log = new Log();

    public SecurityManager securityManager = null;

    static Container local = null;

    Container(int aid) {
    	id = aid;
    	threadGroup = new ContainerThreadGroup("blues.Container#"+Integer.toString(aid),this);
        actionThreads = (ThreadPoolExecutor)java.util.concurrent.Executors.newCachedThreadPool();
    	actionThreads.setThreadFactory(threadGroup);
    }

    public int id() {
        return id;
    }

    public static Container my() {

        ThreadGroup g = Thread.currentThread().getThreadGroup();

        while(g!=null) {
            if (g instanceof ContainerThreadGroup) {
                return ((ContainerThreadGroup)g).container;
            }
            g = parentThreadGroup(g);
        }

        return null;
    }

    private static ThreadGroup parentThreadGroup(final ThreadGroup g) {
        return (ThreadGroup) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                return g.getParent();
            }
        });
    }

    public static Container getContainer(ClassLoader cl) {
        while(null!=cl) {
            if (cl instanceof ClientClassLoader) {
                return ((ClientClassLoader)cl).container;
            }
            cl = cl.getParent();
        }
        return null;
    }


    public AccessibleClassLoader getClassLoader() {
        return clientClassLoader;
    }

    public ClassCodeTransformer getTransformer(String name) {
        return transformers.get(name);
    }

    public void addTransformer(String name, ClassCodeTransformer transformer) {
        transformers.put(name, transformer);
    }

    public void execAction(final String actionName,final Blob action, final Action handler) {
        actionThreads.execute(new Runnable(){
            public void run() {
                Blob response;
                try {
                    response = actionInvoke(actionName,action);
                } catch (Throwable e) {
                    response = Server.exceptionResponse(e,action.id);
                }
                if (null != handler) {
                    try {
                        handler.action(response);
                    } catch (Exception e) {
                    }
                }
            }
        });
    }

    private synchronized Action findAction(String name) throws Exception {
        Action action = actions.get(name);
        if (action==null) {
            int ndx = name.lastIndexOf('.');
            final String className = name.substring(0, ndx);
            final String actionName =  name.substring(ndx+1);
            Class cls = clientClassLoader.loadClass(className);
            final Class[] parameterTypes = { String.class };
            Method mth = cls.getMethod("getAction", parameterTypes);
            final Object[] args = { actionName };
            action = (Action) mth.invoke(null, args);
            if (action!=null) {
                actions.put(name, action);
            }
        }
        return action;
    }

    private Blob actionInvoke(String actionName,Blob actionData) throws Exception {
        Action action = findAction(actionName);
        Blob result = action.action(actionData);
        if (null!=result) {
            result.id = actionData.id + Server.ACTION_RESPONSE;
        }
        return result;
    }

    public synchronized void atExit(Closeable object) {
        cleanups.add(object);
    }

    private void runCleanups() {
        synchronized(cleanups) {
            actionThreads.execute(new Runnable(){
                public void run() {
                    for (int i=0;i<cleanups.size();i++) {
                        try {
                            cleanups.get(cleanups.size()-(i+1)).close();
                        } catch (IOException e) {
                        }
                    }
                    synchronized(cleanups) {
                        cleanups.notify();
                    }
                }
            });
            try {
                cleanups.wait();
            } catch (InterruptedException e) {
            }
        }
    }

    public synchronized void close() {

    	if (null!=cleanups) {
    	    runCleanups();
	        cleanups.clear();
	        cleanups = null;
    	}

    	if (null!=actionThreads) {
	        actionThreads.shutdownNow();
	        try {
				actionThreads.awaitTermination(2,TimeUnit.SECONDS);
			} catch (InterruptedException e) {
		        log.log0(-1,Log.LOG_ERROR, "Container termination timeout: ", e.toString());
			}
	        actionThreads = null;
    	}

        log.close();
    }

    private static File tempDir = null;
    private static long counter = 0;

    private static File createTempDir() {
        File tmp;
        try {
            tmp = File.createTempFile("blueslocal", null, null);
            tmp.delete();
            tmp.mkdirs();
            tmp.deleteOnExit();
        } catch (IOException e) {
            tmp = null;
        }
        return tmp;
    }

    private static void initTempDir() {
        if (tempDir == null) {
            tempDir = (File) AccessController.doPrivileged(new PrivilegedAction(){
                public Object run() {
                    return createTempDir();
                }
            });
        }
    }

    public static String getTempDir() {
        initTempDir();
        return tempDir.getAbsolutePath();
    }

    public static synchronized File getTempName(String name) {
        initTempDir();
        if (tempDir == null) {
            return null;
        }

        String suffix = ".tmp";
        int ldot = name.lastIndexOf('.');
        if (ldot > 0) {
            suffix = name.substring(ldot);
        }
        final String tname = Long.toHexString(counter++);
        StringBuilder sb = new StringBuilder(8 + tname.length() + 4);
        if (tname.length() < 8) {
            for (int i = 0; i < (8 - tname.length()); i++) {
                sb.append('0');
            }
        }
        sb.append(tname);
        sb.append(suffix);

        final File fileParent = tempDir;
        final String fileName = sb.toString();

        File tf  = (File) AccessController.doPrivileged(new PrivilegedAction(){
            public Object run() {
                File f = new File(fileParent, fileName);
                f.deleteOnExit();
                return f;
            }
        });
        return tf;
    }

    static void cleanupTempDir() {
        AccessController.doPrivileged(new PrivilegedAction(){
            public Object run() {
                cleanupTempDir0();
                return null;
            }
        });
    }

    static void cleanupTempDir0() {
        if (tempDir==null) return;
        Log.log(Log.LOG_FILE, "Cleaning up local temp dir ",tempDir.getAbsolutePath());

        File[] files = tempDir.listFiles();
        if (files!=null) {
            for (File f : files) {
                f = f.getAbsoluteFile();
                if (f.delete()==false) {
                    Log.log(Log.LOG_FILE,"Unable to delete local file ",f.getAbsolutePath());
                }
            }
        }
        if (tempDir.delete()==false) {
            Log.log(Log.LOG_FILE,"Unable to delete local temp dir ",tempDir.getAbsolutePath());
        }
        tempDir = null;
    }

}
