/*
 * This file is part of libbluray
 * Copyright (C) 2010-2017  Petri Hintukainen <phintuka@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package hdmv;

import hdmv.INSN.HDMV_INSN;
import hdmv.INSN.MOBJ_CMD;
import net.java.bd.tools.movieobject.MovieObject;
import net.java.bd.tools.movieobject.MovieObjects;

import java.util.ArrayList;
import java.util.List;

import org.videolan.Libbluray;

import blues.Log;
import blues.libbluray.LB;

public class VM {

    static class NV_TIMER {
        long    time;
        int     mobj_id;
    }

    public static class HDMV_EVENT {
    }

    public static class HDMV_EVENT_PLAY_STOP extends HDMV_EVENT {
    }

    public static class HDMV_EVENT_TITLE  extends HDMV_EVENT {
        public int title;
    }

    public static class HDMV_EVENT_PLAY_PL  extends HDMV_EVENT {
        public int playlist;
        public int playitem;
        public int playmark;
    }


    /* state */
    int            pc;            /* program counter */
    MovieObject    object;        /* currently running object code */
    List<HDMV_EVENT> event =      /* pending events to return */
            new ArrayList<HDMV_EVENT>(5);

    NV_TIMER       nv_timer;      /* navigation timer */
    long           rand;          /* RAND state */

    /* movie objects */
    MovieObjects   movie_objects; /* disc movie objects */
    MovieObject    ig_object;     /* current object from IG stream */

    /* object currently playing playlist */
    MovieObject    playing_object;
    int            playing_pc;

    /* suspended object */
    MovieObject    suspended_object;
    int            suspended_pc;

    /* Available titles. Used to validate CALL_TITLE/JUMP_TITLE. */
    static final boolean  have_top_menu=true;
    static final boolean  have_first_play=true;
    int      num_titles;

/*
 * registers: PSR and GPR access
 */

static final int PSR_FLAG = 0x80000000;

static boolean _is_valid_reg(int reg)
{
    if (0!=(reg & PSR_FLAG)) {
        if (0!=(reg & ~0x8000007f)) {
            return false;
        }
    }  else {
        if (0!=(reg & ~0x00000fff)) {
            return false;
        }
    }
    return true;
}

int _store_reg(int reg, int val)
{
    if (!_is_valid_reg(reg)) {
        BD_DEBUG(DBG_HDMV, "_store_reg(): invalid register 0x%x\n", reg);
        return -1;
    }

    if (0!=(reg & PSR_FLAG)) {
        BD_DEBUG(DBG_HDMV, "_store_reg(): storing to PSR is not allowed\n");
        return -1;
    }  else {
        return REG.bd_gpr_write(reg, val);
    }
}

int _read_reg(int reg)
{
    if (!_is_valid_reg(reg)) {
        BD_DEBUG(DBG_HDMV, "_read_reg(): invalid register 0x%x\n", reg);
        return 0;
    }

    if (0!=(reg & PSR_FLAG)) {
        return REG.bd_psr_read(reg & 0x7f);
    } else {
        return REG.bd_gpr_read(reg);
    }
}

int _read_setstream_regs(int val)
{
    int flags = val & 0xf000f000;
    int reg0 = val & 0xfff;
    int reg1 = (val >> 16) & 0xfff;

    int val0 = REG.bd_gpr_read(reg0) & 0x0fff;
    int val1 = REG.bd_gpr_read(reg1) & 0x0fff;

    return flags | val0 | (val1 << 16);
}

int _read_setbuttonpage_reg(int val)
{
    int flags = val & 0xc0000000;
    int reg0  = val & 0x00000fff;

    int val0  = REG.bd_gpr_read(reg0) & 0x3fffffff;

    return flags | val0;
}

int _store_result(MOBJ_CMD cmd, int src, int dst, int src0, int dst0)
{
    int ret = 0;

    /* store result to destination register(s) */
    if (dst != dst0) {
        if (cmd.insn.imm_op1) {
            BD_DEBUG(DBG_HDMV|DBG_CRIT, "storing to imm !\n");
            return -1;
        }
        ret = _store_reg(cmd.dst, dst);
    }

    if (src != src0) {
        if (cmd.insn.imm_op1) {
            BD_DEBUG(DBG_HDMV|DBG_CRIT, "storing to imm !\n");
            return -1;
        }
        ret += _store_reg(cmd.src, src);
    }

    return ret;
}

int _fetch_operand(boolean setstream, boolean setbuttonpage, boolean imm, int value)
{
    if (imm) {
        return value;
    }

    if (setstream) {
        return _read_setstream_regs(value);

    } else if (setbuttonpage) {
        return _read_setbuttonpage_reg(value);

    } else {
        return _read_reg(value);
    }
}

/*
 * event queue
 */

boolean _get_event() { // "0=false"-have , "-1=true" - none
    return (true==event.isEmpty());
}

public HDMV_EVENT hdmv_vm_get_event() {
    return (event.isEmpty())?null:event.remove(0);
}

void _queue_event_HDMV_EVENT_PLAY_STOP() {
    event.add(new HDMV_EVENT_PLAY_STOP());
}

void _queue_event_HDMV_EVENT_TITLE(int title) {
    HDMV_EVENT_TITLE e = new HDMV_EVENT_TITLE();
    e.title = title;
    event.add(e);
}

void _queue_event_HDMV_EVENT_PLAY_PL(int playlist,int playitem,int playmark) {
    HDMV_EVENT_PLAY_PL e = new HDMV_EVENT_PLAY_PL();
    e.playlist = playlist;
    e.playitem = playitem;
    e.playmark = playmark;
    event.add(e);
}

/*
 * vm init
 */


public void hdmv_vm_init()
{
    /* read movie objects */
    this.movie_objects   = LB.getMovieObjects();

    this.num_titles      = Libbluray.numTitles();
    this.rand = 0xfe10;
    
    event.clear();
}

void _free_ig_object() {
    this.ig_object = null;
}

/*
 * suspend/resume ("function call")
 */

boolean _suspended_at_play_pl()
{
    boolean play_pl = false;
    if (null!=this.suspended_object) {
        MOBJ_CMD  cmd  = this.suspended_object.cmds()[this.suspended_pc];
        HDMV_INSN insn = cmd.insn;
        play_pl = (insn.grp     == INSN.GROUP_BRANCH &&
                   insn.sub_grp == INSN.BRANCH_PLAY  &&
                   (  insn.branch_opt == INSN.PLAY_PL ||
                      insn.branch_opt == INSN.PLAY_PL_PI ||
                      insn.branch_opt == INSN.PLAY_PL_PM));
    }

    return play_pl;
}

int _suspend_for_play_pl()
{
    if (null!=this.playing_object) {
        BD_DEBUG(DBG_HDMV|DBG_CRIT, "_suspend_for_play_pl(): object already playing playlist !\n");
        return -1;
    }

    this.playing_object = this.object;
    this.playing_pc     = this.pc;

    this.object = null;

    return 0;
}

int _resume_from_play_pl()
{
    if (null==this.playing_object) {
        BD_DEBUG(DBG_HDMV|DBG_CRIT, "_resume_from_play_pl(): object not playing playlist !\n");
        return -1;
    }

    this.object = this.playing_object;
    this.pc     = this.playing_pc + 1;

    this.playing_object = null;
    _free_ig_object();

    return 0;
}

void _suspend_object(boolean psr_backup)
{
    BD_DEBUG(DBG_HDMV, "_suspend_object()\n");

    if (null!=this.suspended_object) {
        BD_DEBUG(DBG_HDMV, "_suspend_object: object already suspended !\n");
        // [execute the call, discard old suspended object (10.2.4.2.2)].
    }

    if (psr_backup) {
        REG.bd_psr_save_state();
    }

    if (null!=this.ig_object) {
        if (null==this.playing_object) {
            BD_DEBUG(DBG_HDMV|DBG_CRIT, "_suspend_object: IG object tries to suspend, no playing object !\n");
            return;
        }
        this.suspended_object = this.playing_object;
        this.suspended_pc     = this.playing_pc;

        this.playing_object = null;

    } else {

        if (null!=this.playing_object) {
            BD_DEBUG(DBG_HDMV|DBG_CRIT, "_suspend_object: Movie object tries to suspend, also playing object present !\n");
            return;
        }

        this.suspended_object = this.object;
        this.suspended_pc     = this.pc;

    }

    this.object = null;

    _free_ig_object();
}

int _resume_object(boolean psr_restore)
{
    if (null==this.suspended_object) {
        BD_DEBUG(DBG_HDMV|DBG_CRIT, "_resume_object: no suspended object!\n");
        return -1;
    }

    this.object = null;
    this.playing_object = null;
    _free_ig_object();

    if (psr_restore) {
        /* check if suspended in play_pl */
        if (_suspended_at_play_pl()) {
            BD_DEBUG(DBG_HDMV, "resuming playlist playback\n");
            this.playing_object = this.suspended_object;
            this.playing_pc     = this.suspended_pc;
            this.suspended_object = null;
            REG.bd_psr_restore_state();

            return 0;
        }
        REG.bd_psr_restore_state();
    }

    this.object = this.suspended_object;
    this.pc     = this.suspended_pc + 1;

    this.suspended_object = null;

    BD_DEBUG(DBG_HDMV, "resuming object %d at %d\n",
             this.object.id(),
             this.pc);

    _queue_event_HDMV_EVENT_PLAY_STOP();

    return 0;
}


/*
 * branching
 */

boolean _is_valid_title(int title)
{
    if (title == 0) {
        return this.have_top_menu;
    }
    if (title == 0xffff) {
        return this.have_first_play;
    }

    return (title > 0 && title <= this.num_titles);
}

int _jump_object(int object)
{
    if (object >= this.movie_objects.num_objects()) {
        BD_DEBUG(DBG_HDMV|DBG_CRIT, "_jump_object(): invalid object %d\n", object);
        return -1;
    }

    BD_DEBUG(DBG_HDMV, "_jump_object(): jumping to object %d\n", object);

    _queue_event_HDMV_EVENT_PLAY_STOP();

    _free_ig_object();
    
    this.playing_object = null;

    this.pc     = 0;
    this.object = this.movie_objects.getMovieObject()[object];

    /* suspended object is not discarded */

    return 0;
}

int _jump_title(int title)
{
    if (_is_valid_title(title)) {
        BD_DEBUG(DBG_HDMV, "_jump_title(%d)\n", title);

        /* discard suspended object */
        this.suspended_object = null;
        this.playing_object = null;
        REG.bd_psr_reset_backup_registers();

        _queue_event_HDMV_EVENT_TITLE(title);
        return 0;
    }

    BD_DEBUG(DBG_HDMV|DBG_CRIT, "_jump_title(%d): invalid title number\n", title);

    return -1;
}

int _call_object(int object)
{
    if (object >= this.movie_objects.num_objects()) {
        BD_DEBUG(DBG_HDMV|DBG_CRIT, "_call_object(): invalid object %d\n", object);
        return -1;
    }

    BD_DEBUG(DBG_HDMV, "_call_object(%d)\n", object);

    _suspend_object(true);

    return _jump_object(object);
}

int _call_title(int title)
{
    if (_is_valid_title(title)) {
        BD_DEBUG(DBG_HDMV, "_call_title(%d)\n", title);

        _suspend_object(true);

        _queue_event_HDMV_EVENT_TITLE(title);

        return 0;
    }

    BD_DEBUG(DBG_HDMV|DBG_CRIT, "_call_title(%d): invalid title number\n", title);

    return -1;
}

/*
 * playback control
 */

int _play_at(int playlist, int playitem, int playmark)
{
    if ((null!=this.ig_object) && playlist >= 0) {
        BD_DEBUG(DBG_HDMV | DBG_CRIT, "play_at(list %d, item %d, mark %d): " +
              "playlist change not allowed in interactive composition\n",
              playlist, playitem, playmark);
        return -1;
    }

    if ((null==this.ig_object) && playlist < 0) {
        BD_DEBUG(DBG_HDMV | DBG_CRIT, "play_at(list %d, item %d, mark %d): " +
              "playlist not given in movie object (link commands not allowed)\n",
              playlist, playitem, playmark);
        return -1;
    }

    BD_DEBUG(DBG_HDMV, "play_at(list %d, item %d, mark %d)\n",
          playlist, playitem, playmark);

    if (playlist >= 0) {
        _queue_event_HDMV_EVENT_PLAY_PL(playlist,playitem,playmark);
        _suspend_for_play_pl();
    }

    if (playitem >= 0) {
        //_queue_event(p, HDMV_EVENT_PLAY_PI, playitem);
    }

    if (playmark >= 0) {
        //_queue_event(p, HDMV_EVENT_PLAY_PM, playmark);
    }

    return 0;
}

int _play_stop()
{
    if (null==this.ig_object) {
        BD_DEBUG(DBG_HDMV | DBG_CRIT, "_play_stop() not allowed in movie object\n");
        return -1;
    }

    BD_DEBUG(DBG_HDMV, "_play_stop()\n");
    _queue_event_HDMV_EVENT_PLAY_STOP();

    /* terminate IG object. Continue executing movie object.  */
    if (_resume_from_play_pl() < 0) {
        BD_DEBUG(DBG_HDMV|DBG_CRIT, "_play_stop(): resuming movie object failed !\n");
        return -1;
    }

    return 0;
}

/*
 * SET/SYSTEM setstream
 */

void _set_stream(int dst, int src)
{
    BD_DEBUG(DBG_HDMV, "_set_stream(0x%x, 0x%x)\n", dst, src);

    /* primary audio stream */
    if (0!=(dst & 0x80000000)) {
        REG.bd_psr_write(REG.PSR_PRIMARY_AUDIO_ID, (dst >> 16) & 0xfff);
    }

    /* IG stream */
    if (0!=(src & 0x80000000)) {
        REG.bd_psr_write(REG.PSR_IG_STREAM_ID, (src >> 16) & 0xff);
    }

    /* angle number */
    if (0!=(src & 0x8000)) {
        REG.bd_psr_write(REG.PSR_ANGLE_NUMBER, src & 0xff);
    }

    /* PSR2 */
    int psr2 = REG.bd_psr_read(REG.PSR_PG_STREAM);

    /* PG TextST stream number */
    if (0!=(dst & 0x8000)) {
        int text_st_num = dst & 0xfff;
        psr2 = text_st_num | (psr2 & 0xfffff000);
    }

    /* Update PG TextST stream display flag */
    int disp_s_flag = (dst & 0x4000) << 17;
    psr2 = disp_s_flag | (psr2 & 0x7fffffff);

    REG.bd_psr_write(REG.PSR_PG_STREAM, psr2);
}

void _set_sec_stream(int dst, int src)
{
    BD_DEBUG(DBG_HDMV, "_set_sec_stream(0x%x, 0x%x)\n", dst, src);

    int disp_v_flag   = (dst >> 30) & 1;
    int disp_a_flag   = (src >> 30) & 1;
    int text_st_flags = (src >> 13) & 3;

    /* PSR14 */

    int psr14 = REG.bd_psr_read(REG.PSR_SECONDARY_AUDIO_VIDEO);

    /* secondary video */
    if (0!=(dst & 0x80000000)) {
      int sec_video = dst & 0xff;
      psr14 = (sec_video << 8) | (psr14 & 0xffff00ff);
    }

    /* secondary video size */
    if (0!=(dst & 0x00800000)) {
      int video_size = (dst >> 16) & 0xf;
      psr14 = (video_size << 24) | (psr14 & 0xf0ffffff);
    }

    /* secondary audio */
    if (0!=(src & 0x80000000)) {
      int sec_audio = (src >> 16) & 0xff;
      psr14 = sec_audio | (psr14 & 0xffffff00);
    }

    psr14 = (disp_v_flag << 31) | (psr14 & 0x7fffffff);
    psr14 = (disp_a_flag << 30) | (psr14 & 0xbfffffff);

    REG.bd_psr_write(REG.PSR_SECONDARY_AUDIO_VIDEO, psr14);

    /* PSR2 */

    int psr2  = REG.bd_psr_read(REG.PSR_PG_STREAM);

    /* PiP PG TextST stream */
    if (0!=(src & 0x8000)) {
        int stream = src & 0xfff;
        psr2 = (stream << 16) | (psr2 & 0xf000ffff);
    }

    psr2 = (text_st_flags << 30) | (psr2 & 0x3fffffff);

    REG.bd_psr_write(REG.PSR_PG_STREAM, psr2);
}

void _set_stream_ss(int dst, int src)
{
    BD_DEBUG(DBG_HDMV, "_set_stream_ss(0x%x, 0x%x)\n", dst, src);

    if (0==(REG.bd_psr_read(REG.PSR_3D_STATUS) & 1)) {
        BD_DEBUG(DBG_HDMV, "_set_stream_ss ignored (PSR22 indicates 2D mode)\n");
        return;
    }

    BD_DEBUG(DBG_HDMV, "_set_stream_ss(0x%x, 0x%x) unimplemented\n", dst, src);
}

void _setsystem_0x10(int dst, int src)
{
    BD_DEBUG(DBG_HDMV, "_set_psr103(0x%x, 0x%x)\n", dst, src);

    /* just a guess ... */
    //bd_psr_write(p->regs, 104, 0);
    REG.bd_psr_write(103, dst);
}

/*
 * SET/SYSTEM navigation control
 */

void _set_button_page(int dst, int src)
{
    if (null!=this.ig_object) {
        int param;
        param =  (src & 0xc0000000) |        /* page and effects flags */
                ((dst & 0x80000000) >> 2) |  /* button flag */
                ((src & 0x000000ff) << 16) | /* page id */
                 (dst & 0x0000ffff);         /* button id */

        // _queue_event(p, HDMV_EVENT_SET_BUTTON_PAGE, param);

         /* terminate */
         this.pc = 1 << 17;

        return;
    }

    /* selected button */
    if (0!=(dst & 0x80000000)) {
        REG.bd_psr_write(REG.PSR_SELECTED_BUTTON_ID, dst & 0xffff);
    }

    /* active page */
    if (0!=(src & 0x80000000)) {
        REG.bd_psr_write(REG.PSR_MENU_PAGE_ID, src & 0xff);
    }
}

void _enable_button(int dst, boolean enable)
{
    /* not valid in movie objects */
    if (null!=this.ig_object) {
        if (enable) {
            //_queue_event(p, HDMV_EVENT_ENABLE_BUTTON,  dst);
        } else {
            //_queue_event(p, HDMV_EVENT_DISABLE_BUTTON, dst);
        }
    }
}

void _set_still_mode(boolean enable)
{
    /* not valid in movie objects */
    if (null!=this.ig_object) {
        //_queue_event(p, HDMV_EVENT_STILL, enable);
    }
}

void _popup_off()
{
    /* not valid in movie objects */
    if (null!=this.ig_object) {
        //_queue_event(p, HDMV_EVENT_POPUP_OFF, 1);
    }
}

/*
 * SET/SYSTEM 3D mode
 */

void _set_output_mode(int dst)
{
    if ((REG.bd_psr_read(REG.PSR_PROFILE_VERSION) & 0x130240) != 0x130240) {
        BD_DEBUG(DBG_HDMV, "_set_output_mode ignored (not running as profile 5 player)\n");
        return;
    }

    int psr22 = REG.bd_psr_read(REG.PSR_3D_STATUS);

    /* update output mode (bit 0). PSR22 bits 1 and 2 are subtitle alignment (_set_stream_ss()) */
    if (0!=(dst & 1)) {
      psr22 |= 1;
    } else {
      psr22 &= ~1;
    }

    REG.bd_psr_write(REG.PSR_3D_STATUS, psr22);
}

/*
 * navigation timer
 */

void _set_nv_timer(int dst, int src)
{
  int mobj_id = dst & 0xffff;
  int timeout = src & 0xffff;

  if (0==timeout) {
    /* cancel timer */
    this.nv_timer.time = 0;

    REG.bd_psr_write(REG.PSR_NAV_TIMER, 0);

    return;
  }

  /* validate params */
  if (mobj_id >= this.movie_objects.num_objects()) {
      BD_DEBUG(DBG_HDMV|DBG_CRIT, "_set_nv_timer(): invalid object id (%d) !\n", mobj_id);
      return;
  }
  if (timeout > 300) {
      BD_DEBUG(DBG_HDMV|DBG_CRIT, "_set_nv_timer(): invalid timeout (%d) !\n", timeout);
      return;
  }

  BD_DEBUG(DBG_HDMV | DBG_CRIT, "_set_nv_timer(): navigation timer not implemented !\n");

  /* set expiration time */
  this.nv_timer.time = 0x10000;
  this.nv_timer.time += timeout;

  this.nv_timer.mobj_id = mobj_id;

  REG.bd_psr_write(REG.PSR_NAV_TIMER, timeout);
}

/*
 * trace
 */

void _hdmv_trace_cmd(StringBuilder buf,int pc, MOBJ_CMD cmd)
{
    if (Log.enabled(Log.LOG_HDMV)) {
        buf.append(String.format("%04d:  ", pc));

        MOBJPrint.mobj_sprint_cmd(buf, cmd);
    }
}

static void _hdmv_trace_res(StringBuilder buf,int new_src, int new_dst, int orig_src, int orig_dst)
{
    if (Log.enabled(Log.LOG_HDMV)) {
        if (new_dst != orig_dst || new_src != orig_src) {
            buf.append("    :  [");
            if (new_dst != orig_dst) {
                buf.append(String.format(" dst 0x%x <== 0x%x ", orig_dst, new_dst));
            }
            if (new_src != orig_src) {
                buf.append(String.format(" src 0x%x <== 0x%x ", orig_src, new_src));
            }
            buf.append("]");
        }
    }
}

/*
 * interpreter
 */

/*
 * tools
 */

static long u32(int v) {
    long r = v & 0x7fffffff;
    if (v<0) {
        r += 0x80000000L;
    }
    return r;
}

int RAND_u32(int range)
{
    this.rand = this.rand * 6364136223846793005L + 1;

    if (range == 0) {
      BD_DEBUG(DBG_HDMV|DBG_CRIT, "RAND_u32: invalid range (0)\n");
      return 1;
    }

    return ((int)(this.rand >>> 33)) % range + 1;
}

static int ADD_u32(int a, int b)
{
  /* overflow -> saturate */
  long result = u32(a) + u32(b);
  return (result < 0x0ffffffffL) ? (int)result : 0xffffffff;
}

static int MUL_u32(int a, int b)
{
  /* overflow -> saturate */
  long result = u32(a) * u32(b);
  return (result < 0x0ffffffffL) ? (int)result : 0xffffffff;
}

/*
 * _hdmv_step()
 *  - execute next instruction from current program
 */
int _hdmv_step()
{
    MOBJ_CMD   cmd  = this.object.cmds()[this.pc];
    HDMV_INSN  insn = cmd.insn;
    int        src  = 0;
    int        dst  = 0;
    int        inc_pc = 1;

    /* fetch operand values */
    boolean setstream = (insn.grp     == INSN.GROUP_SET &&
            insn.sub_grp == INSN.SET_SETSYSTEM  &&
            (  insn.set_opt == INSN.SET_STREAM ||
               insn.set_opt == INSN.SET_SEC_STREAM));

    boolean setbuttonpage = (insn.grp     == INSN.GROUP_SET &&
                insn.sub_grp == INSN.SET_SETSYSTEM  &&
                insn.set_opt == INSN.SET_BUTTON_PAGE);


    if (insn.op_cnt > 0) {
        dst = _fetch_operand(setstream, setbuttonpage, insn.imm_op1, cmd.dst);
    }

    if (insn.op_cnt > 1) {
        src = _fetch_operand(setstream, setbuttonpage, insn.imm_op2, cmd.src);
    }

    /* trace */
    StringBuilder buf = null;
    if (Log.enabled(Log.LOG_HDMV)) {
        buf = new StringBuilder(384); 
        _hdmv_trace_cmd(buf,this.pc, cmd);
    }

    /* execute */
    switch (insn.grp) {
        case INSN.GROUP_BRANCH:
            switch (insn.sub_grp) {
                case INSN.BRANCH_GOTO:
                    if (insn.op_cnt > 1) {
                        BD_DEBUG(DBG_HDMV|DBG_CRIT, "too many operands in BRANCH/GOTO opcode 0x%08x\n", insn.opcode);
                    }
                    switch (insn.branch_opt) {
                        case INSN.NOP:                      break;
                        case INSN.GOTO:  this.pc   = dst - 1; break;
                        case INSN.BREAK: this.pc   = 1 << 17; break;
                        default:
                            BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown BRANCH/GOTO option %d in opcode 0x%08x\n",
                                     insn.branch_opt, insn.opcode);
                            break;
                    }
                    break;
                case INSN.BRANCH_JUMP:
                    if (insn.op_cnt > 1) {
                        BD_DEBUG(DBG_HDMV|DBG_CRIT, "too many operands in BRANCH/JUMP opcode 0x%08x\n", insn.opcode);
                    }
                    switch (insn.branch_opt) {
                        case INSN.JUMP_TITLE:  _jump_title(dst); break;
                        case INSN.CALL_TITLE:  _call_title(dst); break;
                        case INSN.RESUME:      _resume_object(true);   break;
                        case INSN.JUMP_OBJECT: if (0==_jump_object(dst)) { inc_pc = 0; } break;
                        case INSN.CALL_OBJECT: if (0==_call_object(dst)) { inc_pc = 0; } break;
                        default:
                            BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown BRANCH/JUMP option %d in opcode 0x%08x\n",
                                     insn.branch_opt, insn.opcode);
                            break;
                    }
                    break;
                case INSN.BRANCH_PLAY:
                    switch (insn.branch_opt) {
                        case INSN.PLAY_PL:      _play_at(dst,  -1,  -1); break;
                        case INSN.PLAY_PL_PI:   _play_at(dst, src,  -1); break;
                        case INSN.PLAY_PL_PM:   _play_at(dst,  -1, src); break;
                        case INSN.LINK_PI:      _play_at( -1, dst,  -1); break;
                        case INSN.LINK_MK:      _play_at( -1,  -1, dst); break;
                        case INSN.TERMINATE_PL: if (0==_play_stop()) { inc_pc = 0; } break;
                        default:
                            BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown BRANCH/PLAY option %d in opcode 0x%08x\n",
                                     insn.branch_opt, insn.opcode);
                            break;
                    }
                    break;

                default:
                    BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown BRANCH subgroup %d in opcode 0x%08x\n",
                             insn.sub_grp, insn.opcode);
                    break;
            }
            break; /* INSN_GROUP_BRANCH */

        case INSN.GROUP_CMP:
            if (insn.op_cnt < 2) {
                BD_DEBUG(DBG_HDMV|DBG_CRIT, "missing operand in BRANCH/JUMP opcode 0x%08x\n", insn.opcode);
            }
            boolean r;
            switch (insn.cmp_opt) {
                case INSN.BC: r = (0==(dst & ~src)); break;
                case INSN.EQ: r = (dst == src); break;
                case INSN.NE: r = (dst != src); break;
                case INSN.GE: r = (u32(dst) >= u32(src)); break;
                case INSN.GT: r = (u32(dst) >  u32(src)); break;
                case INSN.LE: r = (u32(dst) <= u32(src)); break;
                case INSN.LT: r = (u32(dst) <  u32(src)); break;
                default:
                    BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown COMPARE option %d in opcode 0x%08x\n",
                             insn.cmp_opt, insn.opcode);
                    r = true;
                    break;
            }
            if (false==r) this.pc += 1;
            break; /* INSN_GROUP_CMP */

        case INSN.GROUP_SET:
            switch (insn.sub_grp) {
                case INSN.SET_SET: {
                    int src0 = src;
                    int dst0 = dst;
                    int tmp;

                    if (insn.op_cnt < 2) {
                        BD_DEBUG(DBG_HDMV|DBG_CRIT, "missing operand in SET/SET opcode 0x%08x\n", insn.opcode);
                    }
                    switch (insn.set_opt) {
                        case INSN.MOVE:   dst  = src;         break;
                        case INSN.SWAP:   tmp=dst; dst=src; src=tmp;  break;
                        case INSN.SUB:    dst  = (u32(dst) > u32(src)) ? (int)(u32(dst) - u32(src)) : 0; break;
                        case INSN.DIV:    dst  = (src != 0) ? (int)(u32(dst) / u32(src)) : 0xffffffff; break;
                        case INSN.MOD:    dst  = (src != 0) ? (int)(u32(dst) % u32(src)) : 0xffffffff; break;
                        case INSN.ADD:    dst  = ADD_u32(src, dst);  break;
                        case INSN.MUL:    dst  = MUL_u32(dst, src);  break;
                        case INSN.RND:    dst  = RAND_u32(src);   break;
                        case INSN.AND:    dst &= src;         break;
                        case INSN.OR:     dst |= src;         break;
                        case INSN.XOR:    dst ^= src;         break;
                        case INSN.BITSET: if (src<32) dst |=  (1 << src); break;
                        case INSN.BITCLR: if (src<32) dst &= ~(1 << src); break;
                        case INSN.SHL:    if (src<32) dst <<= src;        break;
                        case INSN.SHR:    if (src<32) dst >>>= src;       break;
                        default:
                            BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown SET option %d in opcode 0x%08x\n",
                                     insn.set_opt, insn.opcode);
                            break;
                    }

                    /* store result(s) */
                    if (dst != dst0 || src != src0) {

                        _hdmv_trace_res(buf, src, dst, src0, dst0);

                        _store_result(cmd, src, dst, src0, dst0);
                    }
                    break;
                }
                case INSN.SET_SETSYSTEM:
                    switch (insn.set_opt) {
                        case INSN.SET_STREAM:      _set_stream     (dst, src); break;
                        case INSN.SET_SEC_STREAM:  _set_sec_stream (dst, src); break;
                        case INSN.SET_NV_TIMER:    _set_nv_timer   (dst, src); break;
                        case INSN.SET_BUTTON_PAGE: _set_button_page(dst, src); break;
                        case INSN.ENABLE_BUTTON:   _enable_button  (dst,true); break;
                        case INSN.DISABLE_BUTTON:  _enable_button  (dst,false); break;
                        case INSN.POPUP_OFF:       _popup_off      ();         break;
                        case INSN.STILL_ON:        _set_still_mode (true);     break;
                        case INSN.STILL_OFF:       _set_still_mode (false);    break;
                        case INSN.SET_OUTPUT_MODE: _set_output_mode(dst);      break;
                        case INSN.SET_STREAM_SS:   _set_stream_ss  (dst, src); break;
                        case INSN.SETSYSTEM_0x10:  _setsystem_0x10 (dst, src); break;
                        default:
                            BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown SETSYSTEM option %d in opcode 0x%08x\n", insn.set_opt, insn.opcode);
                            break;
                    }
                    break;
                default:
                    BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown SET subgroup %d in opcode 0x%08x\n",
                             insn.sub_grp, insn.opcode);
                    break;
            }
            break; /* INSN_GROUP_SET */

        default:
            BD_DEBUG(DBG_HDMV|DBG_CRIT, "unknown operation group %d in opcode 0x%08x\n",
                     insn.grp, insn.opcode);
            break;
    }

    /* inc program counter to next instruction */
    this.pc += inc_pc;

    if (null!=buf) {
        Log.log(1,Log.LOG_HDMV, buf.toString());
    }

    return 0;
}

/*
 * interface
 */

public int hdmv_vm_select_object(int object)
{
    return _jump_object(object);
}

public boolean hdmv_vm_running()
{
    return (this.object!=null);
}

/*
int hdmv_vm_get_uo_mask(HDMV_VM *p)
{
    uint32_t     mask = 0;
    const MOBJ_OBJECT *o = NULL;

    if (!p) {
        return 0;
    }

    bd_mutex_lock(&p->mutex);

    if ((o = (p->object && !p->ig_object) ? p->object : (p->playing_object ? p->playing_object : p->suspended_object))) {
        mask |= o->menu_call_mask;
        mask |= o->title_search_mask << 1;
    }

    bd_mutex_unlock(&p->mutex);
    return mask;
}
*/

public int hdmv_vm_resume()
{
    int result;

    result = _resume_from_play_pl();

    return result;
}

public int hdmv_vm_suspend_pl()
{
    int result = -1;

    if ((null!=this.object) || (null!=this.ig_object)) {
        BD_DEBUG(DBG_HDMV, "hdmv_vm_suspend_pl(): HDMV VM is still running\n");

    } else if (null==this.playing_object) {
        BD_DEBUG(DBG_HDMV, "hdmv_vm_suspend_pl(): No playing object\n");

    } else if (false==this.playing_object.getTerminalInfo().getResumeIntentionFlag()) {
        BD_DEBUG(DBG_HDMV, "hdmv_vm_suspend_pl(): no resume intention flag\n");

        this.playing_object = null;
        result = 0;

    } else {
        this.suspended_object = this.playing_object;
        this.suspended_pc     = this.playing_pc;

        this.playing_object = null;

        REG.bd_psr_save_state();
        result = 0;
    }

    return result;
}

/* terminate program after MAX_LOOP instructions */
static final int MAX_LOOP = 1000000;

public int hdmv_vm_run()
{
    int max_loop = MAX_LOOP;

    /* pending events ? */
    if (!_get_event()) {
        return 0;
    }

    /* valid program ? */
    if (null==this.object) {
        BD_DEBUG(DBG_HDMV|DBG_CRIT, "hdmv_vm_run(): no object selected\n");
        return -1;
    }

    while (--max_loop > 0) {

        /* suspended ? */
        if (null==this.object) {
            BD_DEBUG(DBG_HDMV, "hdmv_vm_run(): object suspended\n");
            _get_event();
            return 0;
        }

        /* terminated ? */
        if (this.pc >= this.object.cmds().length) {
            BD_DEBUG(DBG_HDMV, "terminated with PC=%d\n", this.pc);
            this.object = null;
            //ev->event = HDMV_EVENT_END;

            if (null!=this.ig_object) {
                //ev->event = HDMV_EVENT_IG_END;
                _free_ig_object();
            }

            return 0;
        }

        /* next instruction */
        if (_hdmv_step() < 0) {
            this.object = null;
            return -1;
        }

        /* events ? */
        if (!_get_event()) {
            return 0;
        }
    }

    BD_DEBUG(DBG_HDMV|DBG_CRIT, "hdmv_vm: infinite program ? terminated after %d instructions.\n", MAX_LOOP);
    this.object = null;
    return -1;
}

    final int DBG_HDMV = 0;
    final int DBG_CRIT = 1;

    static void BD_DEBUG(int t,String msg) {
        dbgout(msg);
    }

    static void BD_DEBUG(int t,String msg,int p0) {
        dbgout(String.format(msg,p0));
    }

    static void BD_DEBUG(int t,String msg,int p0,int p1) {
        dbgout(String.format(msg,p0,p1));
    }

    static void BD_DEBUG(int t,String msg,int p0,int p1,int p2) {
        dbgout(String.format(msg,p0,p1,p2));
    }

    static void dbgout(String msg) {
        Log.logLF(3,Log.LOG_HDMV, msg);
    }

}
