/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2025 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package impl.java.io;

public class FsFD {

    FsFile file;
    long offset;
    boolean write;

    public FsFD(FsFile afile, boolean awrite, boolean append) {
        file = afile;
        afile.incrementUseCount();
        synchronized (file) {
            offset = append ? file.getLength() : 0;
        }
        write = awrite;
    }

    public synchronized void close() {
        file.decrementUseCount();
        file = null;
    }

    public synchronized int readBytes(byte b[], int off, int len) {
        if (len == 0)
            return 0;
        if ((len < 0) || (off < 0))
            return -2;
        if ((off + len) > b.length)
            return -2;

        int r;
        synchronized (file) {
            if (offset > file.getLength()) {
                return -1;
            }

            long avail = file.getLength() - offset;
            if (avail <= 0) {
                return -1;
            }

            if (len > avail) {
                len = (int) avail;
            }

            r = file.readBytes(offset, b, off, len);
        }
        if (r > 0) {
            offset += r;
        }
        return r;
    }

    public synchronized int writeBytes(byte b[], int off, int len) {

        if (false == write)
            return -2;

        int r;
        synchronized (file) {
            if ((file.getBooleanAttributes() & FsObject.ACCESS_WRITE) == 0) {
                return -1;
            }

            if (len == 0)
                return 0;
            if ((len < 0) || (off < 0))
                return -2;
            if ((off + len) > b.length)
                return -2;

            r = file.writeBytes(offset, b, off, len);
        }
        if (r > 0) {
            offset += r;
        }
        return r;
    }

    public synchronized long seek(long pos, boolean relative) {
        if (pos < 0)
            return -2;

        long newPos = pos;

        if (relative) {
            newPos += offset;
        }

        synchronized (file) {
            if (newPos > file.getLength()) {
                newPos = file.getLength();
            }
        }

        offset = newPos;

        return offset;
    }

    public synchronized int setLength(long newLength) {
        if (newLength < 0)
            return -2;

        if (false == write)
            return -1;

        synchronized (file) {
            if ((file.getBooleanAttributes() & FsObject.ACCESS_WRITE) == 0) {
                return -1;
            }

            file.setLength(newLength);
        }

        if (offset > newLength) {
            offset = newLength;
        }

        return 0;
    }

    public synchronized long getLength() {
        synchronized (file) {
            return file.getLength();
        }
    }

    public synchronized int available() {
        synchronized (file) {
            long avail = file.getLength() - offset;
            if (avail < 0)
                avail = 0;
            if (avail > Integer.MAX_VALUE) {
                avail = Integer.MAX_VALUE;
            }
            return (int) avail;
        }
    }
}
